/* SEO Buddy — Admin JavaScript */

jQuery( document ).ready( function( $ ) {
	'use strict';

	var pollInterval = null;

	/**
	 * Validate the license key.
	 */
	function validateKey() {
		var $btn    = $( '#seobuddy-validate-btn' );
		var $input  = $( '#seobuddy-license-key' );
		var $status = $( '#seobuddy-key-status' );
		var key     = $.trim( $input.val() );

		if ( ! key ) {
			showStatus( $status, false, 'Please enter a license key.' );
			return;
		}

		$btn.prop( 'disabled', true ).text( 'Validating...' );
		$status.hide();

		$.post( seobuddy.ajaxurl, {
			action:      'seobuddy_validate_key',
			_ajax_nonce: seobuddy.nonce,
			license_key: key
		} )
		.done( function( response ) {
			if ( response.success && response.data.valid ) {
				var msg = 'Valid';
				if ( response.data.email ) {
					msg += ' — ' + response.data.email;
				}
				showStatus( $status, true, msg );
			} else {
				var errMsg = ( response.data && response.data.message ) ? response.data.message : 'Invalid license key.';
				showStatus( $status, false, errMsg );
			}
		} )
		.fail( function() {
			showStatus( $status, false, 'Connection error. Please try again.' );
		} )
		.always( function() {
			$btn.prop( 'disabled', false ).text( 'Validate' );
		} );
	}

	/**
	 * Show status indicator.
	 */
	function showStatus( $el, valid, text ) {
		var dotClass = valid ? 'valid' : 'invalid';
		$el.find( '.seobuddy-status-dot' ).removeClass( 'valid invalid' ).addClass( dotClass );
		$el.find( '.seobuddy-status-text' ).text( text );
		$el.show();
	}

	/**
	 * Run a full audit.
	 */
	function runAudit() {
		var $btn      = $( '#seobuddy-run-audit-btn' );
		var $progress = $( '#seobuddy-progress' );
		var $report   = $( '#seobuddy-report-container' );

		$btn.prop( 'disabled', true ).text( 'Starting...' );
		$progress.show().find( '.seobuddy-progress-text' ).text( 'Starting audit...' );
		$report.hide();

		$.post( seobuddy.ajaxurl, {
			action:      'seobuddy_run_audit',
			_ajax_nonce: seobuddy.nonce
		} )
		.done( function( response ) {
			if ( response.success && response.data.audit_id ) {
				$btn.text( 'Audit Running...' );
				$progress.find( '.seobuddy-progress-text' ).text( 'Audit in progress. Checking status every 30 seconds...' );
				startPolling( response.data.audit_id );
			} else {
				var errMsg = ( response.data && response.data.message ) ? response.data.message : 'Failed to start audit.';
				$progress.hide();
				showError( errMsg );
				$btn.prop( 'disabled', false ).text( 'Run Full Audit' );
			}
		} )
		.fail( function() {
			$progress.hide();
			showError( 'Connection error. Please try again.' );
			$btn.prop( 'disabled', false ).text( 'Run Full Audit' );
		} );
	}

	/**
	 * Start polling for audit results.
	 */
	function startPolling( auditId ) {
		// Poll immediately, then every 30 seconds.
		pollAudit( auditId );
		pollInterval = setInterval( function() {
			pollAudit( auditId );
		}, 30000 );
	}

	/**
	 * Poll audit status.
	 */
	function pollAudit( auditId ) {
		$.post( seobuddy.ajaxurl, {
			action:      'seobuddy_poll_audit',
			_ajax_nonce: seobuddy.nonce,
			audit_id:    auditId
		} )
		.done( function( response ) {
			if ( ! response.success ) {
				return;
			}

			var data   = response.data;
			var status = data.status || '';

			// Update progress text with current status.
			var $progressText = $( '#seobuddy-progress .seobuddy-progress-text' );

			if ( 'complete' === status ) {
				stopPolling();
				$( '#seobuddy-progress' ).hide();
				$( '#seobuddy-run-audit-btn' ).prop( 'disabled', false ).text( 'Run Full Audit' );
				renderReport( data );
			} else if ( 'failed' === status || 'error' === status ) {
				stopPolling();
				$( '#seobuddy-progress' ).hide();
				$( '#seobuddy-run-audit-btn' ).prop( 'disabled', false ).text( 'Run Full Audit' );
				showError( 'Audit failed. Please try again.' );
			} else {
				// Still processing.
				var statusLabel = status.charAt( 0 ).toUpperCase() + status.slice( 1 );
				$progressText.text( 'Status: ' + statusLabel + '. Checking again in 30 seconds...' );
			}
		} )
		.fail( function() {
			// Silently continue polling on transient errors.
		} );
	}

	/**
	 * Stop polling.
	 */
	function stopPolling() {
		if ( pollInterval ) {
			clearInterval( pollInterval );
			pollInterval = null;
		}
	}

	/**
	 * Render the audit report.
	 */
	function renderReport( data ) {
		var $container = $( '#seobuddy-report-container' );
		var $score     = $( '#seobuddy-score' );
		var $report    = $( '#seobuddy-report' );

		// Render score.
		var score      = parseInt( data.score, 10 ) || 0;
		var scoreClass = 'score-poor';
		if ( score >= 80 ) {
			scoreClass = 'score-good';
		} else if ( score >= 50 ) {
			scoreClass = 'score-ok';
		}

		$score.html(
			'<div class="seobuddy-score-number ' + scoreClass + '">' + score + '</div>' +
			'<span class="seobuddy-score-label">SEO Score</span>'
		);

		// Render markdown report.
		var markdown = data.report_markdown || data.report || '';
		if ( markdown && typeof marked !== 'undefined' ) {
			$report.html( marked.parse( markdown ) );
		} else if ( markdown ) {
			// Fallback: render as preformatted text.
			$report.html( '<pre>' + $( '<span>' ).text( markdown ).html() + '</pre>' );
		} else {
			$report.html( '<p>No report content available.</p>' );
		}

		$container.show();

		// Scroll to report.
		$( 'html, body' ).animate( {
			scrollTop: $container.offset().top - 40
		}, 400 );
	}

	/**
	 * Show an error message.
	 */
	function showError( message ) {
		// Remove any existing error.
		$( '.seobuddy-error' ).remove();

		var $error = $( '<div class="seobuddy-error">' ).text( message );
		$( '#seobuddy-run-audit-btn' ).after( $error );

		setTimeout( function() {
			$error.fadeOut( function() {
				$error.remove();
			} );
		}, 8000 );
	}

	/**
	 * Load the last audit report on page load.
	 */
	function loadLastAudit() {
		var $hidden = $( '#seobuddy-last-audit-id' );
		if ( ! $hidden.length ) {
			return;
		}
		var auditId = $hidden.val();
		if ( ! auditId ) {
			return;
		}

		$.post( seobuddy.ajaxurl, {
			action:      'seobuddy_poll_audit',
			_ajax_nonce: seobuddy.nonce,
			audit_id:    auditId
		} )
		.done( function( response ) {
			if ( response.success && 'complete' === response.data.status ) {
				renderReport( response.data );
			}
		} );
	}

	// Event bindings.
	$( '#seobuddy-validate-btn' ).on( 'click', validateKey );
	$( '#seobuddy-run-audit-btn' ).on( 'click', runAudit );

	// Allow Enter key on license input.
	$( '#seobuddy-license-key' ).on( 'keypress', function( e ) {
		if ( 13 === e.which ) {
			e.preventDefault();
			validateKey();
		}
	} );

	// Auto-load last report.
	loadLastAudit();
} );
